<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use App\Services\CoinPriceService;

class StreamCryptoPrices extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'crypto:stream-prices';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Stream cryptocurrency prices from CoinEx WebSocket and Tetherland API';

    /**
     * CoinEx WebSocket URL
     */
    const COINEX_WS_URL = 'ssl://socket.coinex.com:443';

    /**
     * Tetherland API URL
     */
    const TETHERLAND_API = 'https://api.tetherland.com/currencies';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Starting crypto price stream...');
        
        $context = stream_context_create();
        $socket = stream_socket_client(self::COINEX_WS_URL, $errno, $errstr, 30, STREAM_CLIENT_CONNECT, $context);

        if (!$socket) {
            $this->error("Could not connect to CoinEx WebSocket: $errstr ($errno)");
            return 1;
        }

        $this->info('Connected to CoinEx WebSocket.');

        // Handshake
        $key = base64_encode(openssl_random_pseudo_bytes(16));
        $header = "GET /v2/spot HTTP/1.1\r\n";
        $header .= "Host: socket.coinex.com\r\n";
        $header .= "Upgrade: websocket\r\n";
        $header .= "Connection: Upgrade\r\n";
        $header .= "Sec-WebSocket-Key: $key\r\n";
        $header .= "Sec-WebSocket-Version: 13\r\n";
        $header .= "Sec-WebSocket-Extensions: \r\n\r\n"; // Request no compression

        fwrite($socket, $header);
        $response = fread($socket, 2048);

        if (strpos($response, ' 101 ') === false) {
            $this->error("WebSocket handshake failed.");
            return 1;
        }

        $this->info('Handshake successful.');

        // Subscribe to all tickers
        // CoinEx V2 Spot Ticker Subscribe
        // {"method": "state.subscribe", "params": {"market_list": []}, "id": 1} 
        // Empty market_list means all? No, usually need to specify. 
        // Let's try to get top coins or verify documentation. 
        // User said: wss://socket.coinex.com/v2/spot
        // Based on CoinEx V2 docs, we might need to subscribe to specific markets.
        // For now, let's subscribe to major ones to test, or try to find a way to get all.
        // Actually, CoinEx V1 allowed "ticker.subscribe" for all. V2 might be stricter.
        // Let's try subscribing to a few major ones first.
        
        // Get all symbols from DB to subscribe
        $cryptocurrencies = \App\Cryptocurrency::all();
        $markets = [];
        foreach ($cryptocurrencies as $crypto) {
            if ($crypto->symbol !== 'USDT') {
                $markets[] = $crypto->symbol . 'USDT';
            }
        }
        
        // CoinEx V2 Subscribe format - subscribe to all markets with empty array
        $subscribeMsg = json_encode([
            "method" => "state.subscribe",
            "params" => [
                "market_list" => [] // Empty array subscribes to ALL markets
            ],
            "id" => 1
        ]);
        
        $this->sendWebSocketMessage($socket, $subscribeMsg);
        $this->info('Subscribed to ALL market tickers via CoinEx V2 WebSocket');

        // Non-blocking read
        stream_set_blocking($socket, 0);

        $lastTetherlandUpdate = 0;
        $tetherlandInterval = 10; // Update Tetherland every 10 seconds
        
        $prices = Cache::get(CoinPriceService::CACHE_KEY, []);
        if (empty($prices)) {
             // Initialize with current service logic if empty
             $service = new CoinPriceService();
             $prices = $service->getAllPrices();
        }

        while (true) {
            // 1. Read from WebSocket
            $data = $this->receiveWebSocketMessage($socket);
            if ($data) {
                $json = json_decode($data, true);
                
                // Debug: Log received data
                $this->info('Received: ' . substr($data, 0, 200));
                
                // Handle Ping/Pong if needed (CoinEx sends {"method": "server.ping", ...})
                if (isset($json['method']) && $json['method'] === 'server.ping') {
                    $pong = json_encode(["method" => "server.pong", "params" => [], "id" => $json['id']]);
                    $this->sendWebSocketMessage($socket, $pong);
                    $this->info('Pong sent');
                    continue;
                }
                
                // Handle State Update (Ticker)
                // V2 response: {"method": "state.update", "data": {"state_list": [{"market": "BTCUSDT", "last": "...", "open": "...", ...}]}}
                if (isset($json['method']) && $json['method'] === 'state.update' && isset($json['data']['state_list'])) {
                    foreach ($json['data']['state_list'] as $ticker) {
                        $market = $ticker['market'];
                        if (substr($market, -4) === 'USDT') {
                            $symbol = substr($market, 0, -4);
                            
                            // Calculate 24h change percentage
                            $change24h = 0;
                            if (isset($ticker['open']) && (float)$ticker['open'] > 0) {
                                $open = (float)$ticker['open'];
                                $last = (float)$ticker['last'];
                                $change24h = (($last - $open) / $open) * 100;
                            }
                            
                            // Get USDT price
                            $usdtBuy = $prices['USDT']['buy'] ?? 117100; // Use Tetherland price or default
                            $usdtSell = $prices['USDT']['sell'] ?? 117100;
                            
                            // Update or create price entry
                            $prices[$symbol] = [
                                'last' => round($usdtBuy * (float)$ticker['last']),
                                'buy' => round($usdtBuy * (float)$ticker['last']),
                                'sell' => round($usdtSell * (float)$ticker['last']),
                                'change_24h' => round($change24h, 2),
                                'high' => isset($ticker['high']) ? round($usdtBuy * (float)$ticker['high']) : 0,
                                'low' => isset($ticker['low']) ? round($usdtBuy * (float)$ticker['low']) : 0,
                                'volume' => isset($ticker['volume']) ? (float)$ticker['volume'] : 0,
                                'usdt_price' => (float)$ticker['last']
                            ];
                        }
                    }
                    
                    // Update Cache
                    Cache::put(CoinPriceService::CACHE_KEY, $prices, 60); // Keep alive for 60s
                    $this->info('Prices updated from WebSocket: ' . count($prices) . ' coins');
                }
            }

            // 2. Poll Tetherland
            if (time() - $lastTetherlandUpdate > $tetherlandInterval) {
                $usdtPrice = $this->fetchTetherlandPrice();
                if ($usdtPrice) {
                    $prices['USDT']['buy'] = $usdtPrice;
                    $prices['USDT']['sell'] = $usdtPrice; // Assuming buy=sell for simplicity or fetch both if avail
                    $prices['USDT']['last'] = $usdtPrice;
                    
                    // Update all IRT prices based on new USDT price
                    foreach ($prices as $symbol => $data) {
                        if ($symbol !== 'USDT' && isset($data['usdt_price'])) {
                            $prices[$symbol]['buy'] = round($usdtPrice * $data['usdt_price']);
                            $prices[$symbol]['sell'] = round($usdtPrice * $data['usdt_price']); // Adjust logic if buy/sell diff needed
                        }
                    }
                    
                    Cache::put(CoinPriceService::CACHE_KEY, $prices, 60);
                    $this->info("Tetherland updated: $usdtPrice");
                }
                $lastTetherlandUpdate = time();
            }

            // Prevent CPU hogging
            usleep(100000); // 100ms
        }

        return 0;
    }

    private function sendWebSocketMessage($socket, $message) {
        $frame = [];
        $frame[0] = 0x81; // Text frame
        $len = strlen($message);

        if ($len <= 125) {
            $frame[1] = $len + 128; // Masked
        } else if ($len <= 65535) {
            $frame[1] = 126 + 128;
            $frame[2] = ($len >> 8) & 0xFF;
            $frame[3] = $len & 0xFF;
        } else {
            $frame[1] = 127 + 128;
            $frame[2] = ($len >> 56) & 0xFF;
            $frame[3] = ($len >> 48) & 0xFF;
            $frame[4] = ($len >> 40) & 0xFF;
            $frame[5] = ($len >> 32) & 0xFF;
            $frame[6] = ($len >> 24) & 0xFF;
            $frame[7] = ($len >> 16) & 0xFF;
            $frame[8] = ($len >> 8) & 0xFF;
            $frame[9] = $len & 0xFF;
        }

        $mask = openssl_random_pseudo_bytes(4);
        $frame = array_merge($frame, array_values(unpack('C*', $mask)));

        for ($i = 0; $i < $len; $i++) {
            $frame[] = ord($message[$i]) ^ ord($mask[$i % 4]);
        }

        fwrite($socket, call_user_func_array('pack', array_merge(['C*'], $frame)));
    }

    private function receiveWebSocketMessage($socket) {
        $header = fread($socket, 2);
        if (!$header || strlen($header) < 2) return null;

        $opcode = ord($header[0]) & 0x0F;
        $len = ord($header[1]) & 0x7F;

        if ($len === 126) {
            $extended = fread($socket, 2);
            if (!$extended || strlen($extended) < 2) return null;
            $len = (ord($extended[0]) << 8) | ord($extended[1]);
        } else if ($len === 127) {
            $extended = fread($socket, 8);
            if (!$extended || strlen($extended) < 8) return null;
            // 64-bit length not fully supported in this simple implementation
            $len = (ord($extended[6]) << 8) | ord($extended[7]); 
        }

        // Masking bit (server to client usually not masked, but check standard)
        // Server to client frames are NOT masked.

        $payload = '';
        if ($len > 0) {
            $payload = fread($socket, $len);
        }
        
        // Handle Ping (Opcode 0x9)
        if ($opcode === 0x9) {
            // Send Pong
            return null; 
        }

        // Handle Close (Opcode 0x8)
        if ($opcode === 0x8) {
            return null;
        }

        // Decompress if data is compressed (CoinEx sends compressed data)
        if ($payload && strlen($payload) > 0) {
            // zlib_decode can auto-detect gzip, deflate, or raw deflate
            $decompressed = @zlib_decode($payload);
            if ($decompressed !== false && $decompressed !== null) {
                return $decompressed;
            }
        }

        return $payload;
    }

    protected function fetchTetherlandPrice()
    {
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, self::TETHERLAND_API);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
            
            $response = curl_exec($ch);
            curl_close($ch);
            
            $data = json_decode($response, true);
            if (isset($data['data']['currencies']['USDT']['price'])) {
                return (float) $data['data']['currencies']['USDT']['price'];
            }
            return null;
        } catch (\Exception $e) {
            return null;
        }
    }
}
