<?php

namespace App\Http\Controllers\Admin;

use App\Orders;
use App\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Auth;
use App\Ticket;
use App\TicketMessage;
use DB;
use Morilog\Jalali;
use App\functions;
use SoulDoit\DataTable\SSP;


class TicketController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if(Auth::user()->access_ticket ==1 || Auth::user()->role == 'admin')
                return $next($request);
            else
                abort(404);
        });
    }

    function index()
    {
        $result = (object) array();
        $result->total = Ticket::count();
        $result->open = Ticket::where('status','!=',3)->count();
        $result->pending = Ticket::where('status',0)->count();
        $result->totalMessage = TicketMessage::count();

        $functions = new \App\functions;
        $tickets = \App\Ticket::select('ticket.id','ticket.title','ticket.unit','ticket.created_at','ticket.status','ticket.seen_admin','ticket.updated_at','users.name','users.family','users.mobile')
            ->leftJoin('users', 'ticket.id_user', '=', 'users.id')
            ->latest('updated_at')
            ->paginate(10);
        foreach ($tickets as $ticket) {
            $ticket->created_at = Jalali\CalendarUtils::strftime('H:i Y/m/d', strtotime($ticket->created_at));
            $ticket->updated_at = Jalali\CalendarUtils::strftime('H:i Y/m/d', strtotime($ticket->updated_at));
        }

        return view('admin.ticket', compact(['result', 'tickets']));
    }

    function insert(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'subject' => 'required',
            'unit' => 'required',
            'message' => 'required',
            'mobile' => 'required|numeric',
            'file' => 'nullable|mimes:jpg,jpeg,png,pdf,doc,,docx,,zip,rar|max:20500',
        ], [
            'subject.required' => 'موضوع را مشخص کنید',
            'mobile.required' => 'موبایل را مشخص کنید',
            'unit.required' => 'واحد را انتخاب کنید',
            'message.required' => 'پیغام خود را درج کنید',
            'file.mimes' => 'فرمت فایل مناسب نیست',
            'file.max' => 'حداکثر مجاز حجم فایل 20 مگابایت است',
        ]);

        if ($validator->fails()) {
            $result = array('status' => false, 'message' => $validator->errors()->first());
            return response()->json($result);
        }

        $user = User::where('mobile',$request->mobile)->first();
        if (!isset($user)) {
            $result = array('status' => false, 'message' => 'موبایل کاربر اشتباه است!');
            return response()->json($result);
        }

        DB::beginTransaction();
        try {
            $ticket = new Ticket;
            $ticket->title = $request->subject;
            $ticket->unit = $request->unit;
            $ticket->id_order = $request->order;
            $ticket->id_user = $user->id;
            $ticket->status = 1;
            $ticket->seen_admin = 1;
            $ticket->save();

            $ticket_message = new TicketMessage;
            $ticket_message->id_ticket = $ticket->id;
            $ticket_message->message = $request->message;
            $ticket_message->author = 'admin';
            $ticket_message->save();

            if (isset($request->file)){

                //save image
                $file = $request->file('file');
                $extension = $file->getClientOriginalExtension();
                $just_name = str_replace(" ", "-", pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
                $just_name_hash = str_replace("/", "",bcrypt($just_name  .time()));
                $name = $just_name_hash.'.'.$extension;


                $directory = 'uploads/Users/'. $user->created_at->year.'/'. $user->created_at->month.'/'.  $user->id;
                $file_path = $directory .'/Ticket/'. $name;
                if (!file_exists(($file_path))) {
                    \File::makeDirectory($directory.'/Ticket/', $mode = 0777, true, true);
                    if($request->file('file')->move($directory.'/Ticket/',$name)){
                        $ticket_message->file_link = $directory . '/Ticket/' .$name;
                    }
                } else {
                    if($request->file('file')->move($directory.'/Ticket/',$just_name_hash.time() . '.' . $extension)){
                        $ticket_message->file_link = $directory . '/Ticket/' .$just_name_hash.time() . '.' . $extension;
                    }
                }
                $ticket_message->save();
            }

            DB::commit();
            $result = array('status' => true, 'message' => 'تیکت شما با موفقیت ثبت شد','id'=>$ticket->id);
            return response()->json($result);

        } catch (\Exception $e) {
            DB::rollback();
            $result = array('status' => false, 'message' => 'عملیات دچار مشکل شد' . $e);
        }

        return response()->json($result);
    }


    function singleIndex(Request $request)
    {
        $ticket = Ticket::where('id',$request->id)->first();
        if($ticket){

            $ticket->seen_admin = 1;
            $ticket->timestamps = false;
            $ticket->save();

            $ticket->created = Jalali\CalendarUtils::strftime('Y/m/d H:i', strtotime($ticket->created_at));
            $ticket->updated = Jalali\CalendarUtils::strftime('Y/m/d H:i', strtotime($ticket->updated_at));

            $ticket_message = TicketMessage::where('id_ticket',$ticket->id)->orderBy('created_at')->get();
            foreach($ticket_message as $message){
                if(date('Y-m-d H:i:s') <date('Y-m-d H:i:s',strtotime('+6 day', strtotime($message->created_at))) ) {
                    $message->time = Jalali\Jalalian::forge($message->created_at)->ago();
                }else {
                    $message->time = Jalali\CalendarUtils::strftime('Y/m/d H:i', strtotime($message->created_at));
                    $message->class = 'sans-serif ltr-dir';
                }
                $admin = \App\AdminUser::find($message->id_admin);
                if($admin)
                    $message->admin = $admin->name;
            }

            if(isset($ticket->id_order)){
                $order = Orders::where('id',$ticket->id_order)->first();
                $functions = new functions;
                $detail = $functions->get_order_detail($order->id);

                $order->title =$detail->type .' '. $detail->title;
                $data["order"] = $order;
            }

            $data["user"] = User::select('id','name','family','mobile')->find($ticket->id_user);
            $data["ticket"] = $ticket;
            $data["ticket_message"] = $ticket_message;
        }else
            abort(404);

        $data['pageConfigs'] = [
            'pageHeader' => false,
            'bodyClass' => 'chat-application',
        ];

        return view('admin.ticket-single',$data);
    }

    function single(Request $request)
    {
        $validator = \Validator::make($request->all(), [
            'message' => 'required',
            'file' => 'nullable|mimes:jpg,jpeg,png,pdf,doc,,docx,,zip,rar|max:20500',
        ], [
            'message.required' => 'پیغام خود را درج کنید',
            'file.mimes' => 'فرمت فایل مناسب نیست',
            'file.max' => 'حداکثر مجاز حجم فایل 20 مگابایت است',
        ]);

        if ($validator->fails()) {
            $result = array('status' => false, 'message' => $validator->errors()->first());
            return response()->json($result);
        }

        $ticket = Ticket::where('id',$request->id)->first();
        if($ticket):
            $user = User::where('id',$ticket->id_user)->first();

            DB::beginTransaction();
            try {
                $ticket_message = new TicketMessage;
                $ticket_message->id_ticket = $request->id;
                $ticket_message->message = $request->message;
                $ticket_message->author = 'admin';
                $ticket_message->id_admin = Auth::user()->id;
                $ticket_message->save();

                if (isset($request->file)){

                    //save image
                    $file = $request->file('file');
                    $extension = $file->getClientOriginalExtension();
                    $just_name = str_replace(" ", "-", pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
                    $just_name_hash = str_replace("/", "",bcrypt($just_name  .time()));
                    $name = $just_name_hash.'.'.$extension;


                    $directory = 'uploads/Users/'. $user->created_at->year.'/'. $user->created_at->month.'/'.  $user->id;
                    $file_path = $directory .'/Ticket/'. $name;
                    if (!file_exists(($file_path))) {
                        \File::makeDirectory($directory.'/Ticket/', $mode = 0777, true, true);
                        if($request->file('file')->move($directory.'/Ticket/',$name)){
                            $ticket_message->file_link = $directory . '/Ticket/' .$name;
                        }
                    } else {
                        if($request->file('file')->move($directory.'/Ticket/',$just_name_hash.time() . '.' . $extension)){
                            $ticket_message->file_link = $directory . '/Ticket/' .$just_name_hash.time() . '.' . $extension;
                        }
                    }
                    $ticket_message->save();
                }

                $ticket->seen_user = 0;
                $ticket->status = 1;
                $ticket->save();

                $functions = new \App\functions();
                $functions->LogAdmin('پاسخ به تیکیت کاربر '.$user->mobile.' با عنوان '. $ticket->title,'ticket');


                if (isset($user->email)){
                    $date = Jalali\Jalalian::forge('today')->format('%d %B %Y');
                    $data = array('date'=>'به تیکت شما که با عنوان '.'"'.$ticket->title.'"'.' میباشد پاسخ داده شد.', 'title'=>'', 'body'=>$request->message);
                    /*\Mail::send('emails.notification', $data, function($message) use ($user)
                    {
                        $message->to($user->email, $user->email)->subject('پاسخ به تیکت در '.env('APP_NAME_FARSI'));
                    });*/
                }


                DB::commit();
                $result = array('status' => true, 'message' => 'تیکت شما با موفقیت ثبت شد');
                return response()->json($result);

            } catch (\Exception $e) {
                DB::rollback();
                $result = array('status' => 'false', 'message' => 'عملیات دچار مشکل شد' . $e);
            }
        else:
            $result = array('status' => 'false', 'message' => 'امکان پذیر نیست');
        endif;
        return response()->json($result);
    }


    public function status(Request $request)
    {
        $ticket = Ticket::where('id', $request->id)->first();
        if ($ticket) {
            $ticket->status = $request->status;
            if($request->status == 3){
                $result = array('status' => true , 'messege' => 'با موفقیت بسته شد');

                $user = User::where('id',$ticket->id_user)->first();
                $functions = new \App\functions();
                $functions->LogAdmin('بستن تیکیت کاربر '.$user->mobile.' با عنوان '. $ticket->title,'ticket');
            }
            $ticket->save();
        } else
            $result = array('status' => false , 'messege' => 'خطا در ارسال پارامتر');
        return response()->json($result);
    }



    public function remove(Request $request)
    {
        $ticket = Ticket::where('id', $request->id)->first();
        if ($ticket) {
            DB::beginTransaction();
            try {

                $ticket_message = TicketMessage::where('id_ticket',$ticket->id)->whereNotNull('file_link')->get();
                foreach($ticket_message as $message){
                    if (file_exists(($message->file_link)))
                        unlink($message->file_link);
                }
                TicketMessage::where('id_ticket',$ticket->id)->delete();
                try {
                    if($ticket->delete()){
                        DB::commit();
                        $result = array('status' => true , 'messege' => 'با موفقیت حذف شد');
                    }
                } catch (\Illuminate\Database\QueryException $e) {
                    $result = array('status' => false , 'messege' => 'حذف امکان پذیر نیست');
                }

            } catch (\Exception $e) {
                DB::rollback();
                $result = array('status' => false, 'message' => 'عملیات دچار مشکل شد');
            }
        } else
            $result = array('status' => false , 'messege' => 'خطا در ارسال پارامتر');
        return response()->json($result);
    }


    function mobile_search(Request $request){
        $users = User::select('mobile as data')->where('mobile','like','%'.$request->mobile.'%')->limit(10)->orderBy('id','desc')->get();
        return response()->json($users);
    }

}
