<?php

namespace App;

use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    protected $fillable = [
        'user_id',
        'market',
        'type',
        'side',
        'price',
        'amount',
        'filled_amount',
        'total',
        'status',
        'coinex_order_id',
        'error_message',
    ];

    protected $casts = [
        'price' => 'decimal:8',
        'amount' => 'decimal:8',
        'filled_amount' => 'decimal:8',
        'total' => 'decimal:8',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function trades()
    {
        return $this->hasMany(Trade::class);
    }

    /**
     * Check if order is pending
     */
    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * Check if order is filled
     */
    public function isFilled()
    {
        return $this->status === 'filled';
    }

    /**
     * Check if order is cancelled
     */
    public function isCancelled()
    {
        return $this->status === 'cancelled';
    }

    /**
     * Get remaining amount
     */
    public function getRemainingAmountAttribute()
    {
        return $this->amount - $this->filled_amount;
    }

    /**
     * Update filled amount
     */
    public function updateFilled($filledAmount)
    {
        $this->filled_amount += $filledAmount;
        
        if ($this->filled_amount >= $this->amount) {
            $this->status = 'filled';
        } elseif ($this->filled_amount > 0) {
            $this->status = 'partial';
        }
        
        $this->save();
    }

    /**
     * Cancel order
     */
    public function cancel()
    {
        $this->status = 'cancelled';
        $this->save();
    }
}
