<?php

namespace App\Services;

use App\Cryptocurrency;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class CoinPriceService
{
    /**
     * Cache duration in seconds (1 second for real-time updates)
     */
    const CACHE_DURATION = 1;
    
    /**
     * Cache key for all prices
     */
    const CACHE_KEY = 'crypto_prices_live';
    
    /**
     * CoinEx API endpoint for all tickers
     */
    const API_ENDPOINT = 'https://api.coinex.com/v1/market/ticker/all';
    
    /**
     * Tetherland API endpoint for all tickers
     */
    const TETHERLAND_API = 'https://api.tetherland.com/currencies';

    /**
     * Get all cryptocurrency prices with caching
     *
     * @return array
     */
    public function getAllPrices()
    {
        $prices = null;
        
        // 1. Try JSON file first (Priority: Stream Command)
        // This ensures we get the latest data written by the background process
        // and avoids cache permission issues between CLI (root) and Web (www-data)
        $jsonPath = public_path('prices.json');
        if (file_exists($jsonPath)) {
            // Check if file is fresh (less than 10 seconds old)
            if (time() - filemtime($jsonPath) < 10) {
                $jsonContent = file_get_contents($jsonPath);
                $prices = json_decode($jsonContent, true);
            }
        }
        
        // 2. If JSON not available or stale, try Cache
        if (!$prices) {
            $prices = Cache::get(self::CACHE_KEY);
        }
        
        // 3. If still empty, fetch from API (fallback)
        if (!$prices) {
            $prices = $this->fetchPricesFromAPI();
            // Cache it
            Cache::put(self::CACHE_KEY, $prices, self::CACHE_DURATION);
        }
        
        return $prices;
    }

    /**
     * Get price for a specific cryptocurrency
     *
     * @param string $symbol Cryptocurrency symbol (e.g., 'BTC', 'ETH')
     * @param string $type Price type: 'buy' or 'sell'
     * @return float|null
     */
    public function getPrice($symbol, $type = 'buy')
    {
        $allPrices = $this->getAllPrices();
        
        if (isset($allPrices[$symbol])) {
            return $allPrices[$symbol][$type] ?? null;
        }
        
        return null;
    }

    /**
     * Get full price data for a cryptocurrency
     *
     * @param string $symbol Cryptocurrency symbol
     * @return object|null
     */
    public function getPriceData($symbol)
    {
        $allPrices = $this->getAllPrices();
        
        if (isset($allPrices[$symbol])) {
            return (object) $allPrices[$symbol];
        }
        
        return null;
    }

    /**
     * Force refresh prices (clear cache and fetch new)
     *
     * @return array
     */
    public function refreshPrices()
    {
        Cache::forget(self::CACHE_KEY);
        return $this->getAllPrices();
    }

    /**
     * Fetch prices from CoinEx API (or fallbacks)
     *
     * @return array
     */
    protected function fetchPricesFromAPI()
    {
        try {
            // Fetch USDT/Toman price based on selected source
            $usdtPrice = $this->fetchUSDTPrice();
            
            // Use fetched price if available, otherwise fallback to manual settings
            $usdtBuy = $usdtPrice ?: (float) getSetting('USDT_price_buy') ?: 50000;
            $usdtSell = $usdtPrice ?: (float) getSetting('USDT_price_sell') ?: 49000;
            
            // Log the USDT price source for debugging
            Log::info('CoinPriceService: Using USDT price', [
                'buy' => $usdtBuy,
                'sell' => $usdtSell,
                'source' => $usdtPrice ? 'API' : 'Manual'
            ]);
            
            // 1. Try Nobitex first (Best for Iran servers)
            Log::info('CoinPriceService: Attempting to fetch from Nobitex...');
            $response = $this->fetchFromNobitex();
            
            // 2. If Nobitex fails, try CoinEx V2
            if (!$response) {
                Log::info('CoinPriceService: Nobitex failed, trying CoinEx V2...');
                $response = $this->fetchFromCoinEx();
            }
            
            // 3. If CoinEx fails, try Binance
            if (!$response) {
                Log::warning('CoinPriceService: CoinEx failed, trying Binance...');
                $response = $this->fetchFromBinance();
            }
            
            // 4. If Binance fails, try CoinGecko
            if (!$response) {
                Log::warning('CoinPriceService: Binance failed, trying CoinGecko...');
                $response = $this->fetchFromCoinGecko();
            }
            
            if (!$response) {
                Log::warning('CoinPriceService: All APIs failed, using approximate fallback prices');
                // Use approximate prices as fallback (these are rough estimates)
                $response = [
                    'source' => 'fallback',
                    'data' => [
                        'BTCUSDT' => ['last' => '95000', 'open' => '94000'],
                        'ETHUSDT' => ['last' => '3400', 'open' => '3350'],
                        'XRPUSDT' => ['last' => '2.2', 'open' => '2.15'],
                    ]
                ];
            }
            
            $data = $response;

            // Process tickers based on source format
            $tickers = [];
            
            if (isset($data['source']) && $data['source'] === 'binance') {
                // Process Binance data
                foreach ($data['data'] as $item) {
                    $symbol = $item['symbol'];
                    if (substr($symbol, -4) === 'USDT') {
                        $coinSymbol = substr($symbol, 0, -4);
                        $tickers[$coinSymbol] = [
                            'last' => $item['price'],
                            'change_24h' => 0 
                        ];
                    }
                }
            } elseif (isset($data['source']) && $data['source'] === 'nobitex') {
                // Process Nobitex data - Nobitex returns prices in IRT (Toman)
                // We need to convert to USDT by dividing by USDT/IRT price
                foreach ($data['data'] as $pair => $stats) {
                    // Nobitex pairs are like: btc-rls, eth-rls, etc.
                    if (strpos($pair, '-rls') !== false) {
                        $coinSymbol = strtoupper(str_replace('-rls', '', $pair));
                        // Convert Rials to Toman (divide by 10) then to USDT
                        $priceInToman = (float)$stats['latest'] / 10;
                        $priceInUSDT = $usdtBuy > 0 ? $priceInToman / $usdtBuy : 0;
                        
                        $tickers[$coinSymbol] = [
                            'last' => $priceInUSDT,
                            'change_24h' => $stats['dayChange'] ?? 0
                        ];
                    }
                }
            } elseif (isset($data['source']) && $data['source'] === 'coingecko') {
                // Process CoinGecko data
                foreach ($data['data'] as $item) {
                    $symbol = strtoupper($item['symbol']);
                    $tickers[$symbol] = [
                        'last' => $item['current_price'],
                        'change_24h' => $item['price_change_percentage_24h']
                    ];
                }
            } elseif (isset($data['source']) && $data['source'] === 'fallback') {
                // Process fallback data
                $tickers = $data['data'];
            } elseif (isset($data['source']) && $data['source'] === 'coinex_v2') {
                // Process CoinEx V2 data
                foreach ($data['data'] as $item) {
                    $market = $item['market'];
                    if (substr($market, -4) === 'USDT') {
                        $symbol = substr($market, 0, -4);
                        
                        // Calculate 24h change
                        $change24h = 0;
                        if (isset($item['open']) && (float)$item['open'] > 0) {
                            $open = (float)$item['open'];
                            $last = (float)$item['last'];
                            $change24h = (($last - $open) / $open) * 100;
                        }
                        
                        $tickers[$symbol] = [
                            'last' => $item['last'],
                            'change_24h' => $change24h,
                            'high' => $item['high'],
                            'low' => $item['low'],
                            'vol' => $item['volume']
                        ];
                    }
                }
            } else {
                // Process CoinEx data (legacy V1)
                $tickers = $data['data']['ticker'] ?? [];
            }

            // Get all cryptocurrencies from database
            $cryptocurrencies = Cryptocurrency::all();
            $prices = [];

            foreach ($cryptocurrencies as $crypto) {
                $symbol = $crypto->symbol;
                $tickerKey = $symbol . 'USDT';

                if ($symbol === 'USDT') {
                    // USDT prices from Tetherland
                    $prices[$symbol] = [
                        'buy' => $usdtBuy,
                        'sell' => $usdtSell,
                        'last' => $usdtBuy,
                        'change_24h' => 0,
                        'high' => $usdtBuy,
                        'low' => $usdtSell,
                        'volume' => 0,
                        'usdt_price' => 1
                    ];
                } elseif (isset($tickers[$symbol]) || isset($tickers[$tickerKey])) {
                    // Handle different key formats
                    $ticker = isset($tickers[$symbol]) ? $tickers[$symbol] : $tickers[$tickerKey];
                    
                    // Calculate IRT prices
                    $cryptoPrice = (float) $ticker['last'];
                    $buyPrice = round($usdtBuy * $cryptoPrice);
                    $sellPrice = round($usdtSell * $cryptoPrice);
                    
                    // Calculate 24h change percentage
                    $change24h = 0;
                    if (isset($ticker['change_24h'])) {
                        $change24h = (float) $ticker['change_24h'];
                    } elseif (isset($ticker['open']) && (float)$ticker['open'] > 0) {
                        $open = (float) $ticker['open'];
                        $last = (float) $ticker['last'];
                        $change24h = (($last - $open) / $open) * 100;
                    }

                    $prices[$symbol] = [
                        'buy' => $buyPrice,
                        'sell' => $sellPrice,
                        'last' => $buyPrice,
                        'change_24h' => round($change24h, 2),
                        'high' => isset($ticker['high']) ? round($usdtBuy * (float) $ticker['high']) : 0,
                        'low' => isset($ticker['low']) ? round($usdtBuy * (float) $ticker['low']) : 0,
                        'volume' => isset($ticker['vol']) ? (float) $ticker['vol'] : 0,
                        'usdt_price' => $cryptoPrice
                    ];
                } else {
                    // Cryptocurrency not found in API, use default values
                    $prices[$symbol] = [
                        'buy' => 0,
                        'sell' => 0,
                        'last' => 0,
                        'change_24h' => 0,
                        'high' => 0,
                        'low' => 0,
                        'volume' => 0,
                        'usdt_price' => 0
                    ];
                }
            }

            // Cache the result
            Cache::put(self::CACHE_KEY, $prices, self::CACHE_DURATION);
            
            return $prices;

        } catch (\Exception $e) {
            Log::error('CoinPriceService: Failed to fetch prices from API', [
                'error' => $e->getMessage()
            ]);

            // Return cached prices if available, otherwise empty array
            return Cache::get(self::CACHE_KEY, []);
        }
    }

    /**
     * Fetch prices from CoinEx
     */
    protected function fetchFromCoinEx()
    {
        try {
            // Get all symbols from DB
            $cryptocurrencies = Cryptocurrency::all();
            $markets = [];
            foreach ($cryptocurrencies as $crypto) {
                if ($crypto->symbol !== 'USDT') {
                    $markets[] = $crypto->symbol . 'USDT';
                }
            }
            
            // Fallback to default list if no coins in DB
            if (empty($markets)) {
                $markets = ['BTCUSDT', 'ETHUSDT', 'XRPUSDT', 'DOGEUSDT', 'BNBUSDT', 'ADAUSDT', 'TRXUSDT', 'LTCUSDT', 'BCHUSDT', 'SOLUSDT'];
            }
            
            $marketStr = implode(',', $markets);
            $url = "https://api.coinex.com/v2/spot/ticker?market=" . $marketStr;
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 20); // Increased timeout
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($httpCode !== 200 || $response === false) {
                Log::warning('CoinPriceService: CoinEx API failed', [
                    'status' => $httpCode,
                    'error' => $error,
                    'url' => $url
                ]);
                return null;
            }

            
            $data = json_decode($response, true);
            if (!isset($data['code']) || $data['code'] !== 0 || !isset($data['data'])) {
                Log::warning('CoinPriceService: Invalid CoinEx response format', ['response' => substr($response, 0, 200)]);
                return null;
            }
            
            // Add source tag
            $data['source'] = 'coinex_v2';
            return $data;
        } catch (\Exception $e) {
            Log::error('CoinPriceService: CoinEx exception', ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Fetch prices from Binance
     */
    protected function fetchFromBinance()
    {
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://api.binance.com/api/v3/ticker/price');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($httpCode !== 200 || $response === false) {
                Log::warning('CoinPriceService: Binance API failed', [
                    'status' => $httpCode,
                    'error' => $error
                ]);
                return null;
            }
            
            $data = json_decode($response, true);
            if (!is_array($data)) return null;
            
            return ['source' => 'binance', 'data' => $data];
        } catch (\Exception $e) {
            Log::error('CoinPriceService: Binance exception', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    /**
     * Fetch prices from CoinGecko
     */
    protected function fetchFromCoinGecko()
    {
        try {
            // Top 50 coins to avoid large payload
            $url = 'https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&order=market_cap_desc&per_page=50&page=1&sparkline=false';
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($httpCode !== 200) {
                Log::warning('CoinPriceService: CoinGecko API failed', [
                    'status' => $httpCode,
                    'error' => $error
                ]);
                return null;
            }
            
            $data = json_decode($response, true);
            if (!is_array($data)) return null;
            
            return ['source' => 'coingecko', 'data' => $data];
        } catch (\Exception $e) {
            Log::error('CoinPriceService: CoinGecko exception', ['error' => $e->getMessage()]);
            return null;
        }
    }
    
    /**
     * Fetch prices from Nobitex
     */
    protected function fetchFromNobitex()
    {
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://api.nobitex.ir/market/stats');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($httpCode !== 200) {
                Log::warning('CoinPriceService: Nobitex API failed', [
                    'status' => $httpCode,
                    'error' => $error
                ]);
                return null;
            }
            
            $data = json_decode($response, true);
            if (!isset($data['status']) || $data['status'] !== 'ok' || !isset($data['stats'])) {
                Log::warning('CoinPriceService: Invalid Nobitex response format');
                return null;
            }
            
            return ['source' => 'nobitex', 'data' => $data['stats']];
        } catch (\Exception $e) {
            Log::error('CoinPriceService: Nobitex exception', ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Fetch USDT/Toman price from Tetherland API
     *
     * @return float|null
     */
    protected function fetchTetherlandPrice()
    {
        try {
            // Check if Tetherland is enabled in settings
            $usdt_price_source = getSetting('usdt_price_source');
            if ($usdt_price_source && $usdt_price_source->value !== 'tetherland') {
                return null; // Tetherland not selected as source
            }
            
            // Get API URL and private key from settings
            $apiUrl = getSetting('tetherland_api_url');
            $privateKey = getSetting('tetherland_private_key');
            
            // Use the new endpoint by default, or the one from settings if provided
            $endpoint = ($apiUrl && !empty($apiUrl->value)) ? $apiUrl->value : self::TETHERLAND_API;
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $endpoint);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            
            // Prepare headers
            $headers = [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'Accept: application/json, text/plain, */*',
                'Accept-Language: en-US,en;q=0.9',
                'Referer: https://tetherland.com/',
                'Origin: https://tetherland.com'
            ];
            
            // Add private key to headers if available (though likely not needed for public endpoint)
            if ($privateKey && !empty($privateKey->value)) {
                $key = trim($privateKey->value);
                $headers[] = 'Authorization: Bearer ' . $key;
            }
            
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Increased timeout
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true); // Follow redirects
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            if ($httpCode !== 200) {
                throw new \Exception('Tetherland API returned status: ' . $httpCode . ($curlError ? ' - ' . $curlError : ''));
            }
            
            $data = json_decode($response, true);
            
            // Check for new response structure: data.currencies.USDT.price
            if (isset($data['data']['currencies']['USDT']['price'])) {
                $price = (float) $data['data']['currencies']['USDT']['price'];
                Log::info('CoinPriceService: Fetched USDT price from Tetherland (New API)', ['price' => $price]);
                return $price;
            }
            
            // Fallback to old structure if the API changes back or settings point to old API
            if (isset($data['status']) && $data['status'] === true && isset($data['data']['coinPrice'])) {
                Log::info('CoinPriceService: Fetched USDT price from Tetherland (Old API)', ['price' => $data['data']['coinPrice']]);
                return (float) $data['data']['coinPrice'];
            }
            
            Log::warning('CoinPriceService: Invalid Tetherland response format', ['response' => substr($response, 0, 200)]);
            return null;
            
        } catch (\Exception $e) {
            Log::warning('CoinPriceService: Failed to fetch Tetherland price', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }


    /**
     * Fetch USDT/Toman price based on selected source
     *
     * @return float|null
     */
    protected function fetchUSDTPrice()
    {
        $usdt_price_source = getSetting('usdt_price_source');
        $source = $usdt_price_source ? $usdt_price_source->value : 'tetherland';
        
        switch ($source) {
            case 'tetherland':
                return $this->fetchTetherlandPrice();
            
            case 'iranian_exchange':
                return $this->fetchIranianExchangePrice();
            
            case 'manual':
            default:
                return null; // Will use manual prices from settings
        }
    }

    /**
     * Fetch USDT/Toman price from Iranian Exchange API
     *
     * @return float|null
     */
    protected function fetchIranianExchangePrice()
    {
        try {
            $exchangeName = getSetting('iranian_exchange_name');
            $apiKey = getSetting('iranian_exchange_api_key');
            
            $exchange = $exchangeName ? $exchangeName->value : 'nobitex';
            
            // API endpoints for different Iranian exchanges
            $endpoints = [
                'nobitex' => 'https://api.nobitex.ir/v2/orderbook/USDTIRT',
                'wallex' => 'https://api.wallex.ir/v1/markets',
                'exnovin' => 'https://api.exnovin.com/api/v1/market/stats',
                'ramzinex' => 'https://publicapi.ramzinex.com/exchange/api/v1.0/exchange/pairs'
            ];
            
            if (!isset($endpoints[$exchange])) {
                throw new \Exception('Unknown exchange: ' . $exchange);
            }
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $endpoints[$exchange]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            
            $headers = ['Accept: application/json'];
            if ($apiKey && !empty($apiKey->value)) {
                $headers[] = 'Authorization: Bearer ' . trim($apiKey->value);
            }
            
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_TIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode !== 200) {
                throw new \Exception('Iranian Exchange API returned status: ' . $httpCode);
            }
            
            $data = json_decode($response, true);
            $price = 0;
            
            // Parse price based on exchange
            switch ($exchange) {
                case 'nobitex':
                    if (isset($data['lastTradePrice'])) {
                        $price = (float) $data['lastTradePrice'] / 10; // Nobitex returns in Rials, convert to Toman
                    }
                    break;
                
                case 'wallex':
                    if (isset($data['result']['symbols']['USDTTMN']['stats']['lastPrice'])) {
                        $price = (float) $data['result']['symbols']['USDTTMN']['stats']['lastPrice'];
                    }
                    break;
                
                case 'exnovin':
                    if (isset($data['data']['USDT_IRT']['last'])) {
                        $price = (float) $data['data']['USDT_IRT']['last'] / 10;
                    }
                    break;
                
                case 'ramzinex':
                    // Find USDT pair in response
                    if (isset($data['data'])) {
                        foreach ($data['data'] as $pair) {
                            if (isset($pair['pair']['base_currency_symbol']['symbol']) && 
                                $pair['pair']['base_currency_symbol']['symbol'] === 'usdt') {
                                $price = (float) $pair['sell'] / 10;
                                break;
                            }
                        }
                    }
                    break;
            }
            
            if ($price > 0) {
                Log::info('CoinPriceService: Fetched USDT price from ' . $exchange, ['price' => $price]);
                return $price;
            }
            
            return null;
            
        } catch (\Exception $e) {
            Log::warning('CoinPriceService: Failed to fetch Iranian Exchange price', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Get prices formatted for CryptoController compatibility
     *
     * @param array $symbols Array of cryptocurrency symbols
     * @return object
     */
    public function getPricesForCoins($symbols = [])
    {
        $allPrices = $this->getAllPrices();
        $result = ['coins' => []];

        if (empty($symbols)) {
            // Return all prices
            $cryptocurrencies = Cryptocurrency::all();
            $symbols = $cryptocurrencies->pluck('symbol')->toArray();
        }

        foreach ($symbols as $symbol) {
            if (isset($allPrices[$symbol])) {
                $result['coins'][] = (object) [
                    'symbol' => $symbol,
                    'buy' => $allPrices[$symbol]['buy'],
                    'sell' => $allPrices[$symbol]['sell']
                ];
            }
        }

        return (object) $result;
    }

    /**
     * Get cache status information
     *
     * @return array
     */
    public function getCacheStatus()
    {
        $cached = Cache::has(self::CACHE_KEY);
        $prices = Cache::get(self::CACHE_KEY, []);
        
        return [
            'cached' => $cached,
            'count' => count($prices),
            'ttl' => self::CACHE_DURATION,
            'last_update' => $cached ? now()->subSeconds(self::CACHE_DURATION)->toDateTimeString() : null
        ];
    }
}
